"use strict";

import Globals from "./globals";

class Seo {
    constructor() {
        this.titleSelector = ["title", "name"];
        this.slugSelector = "slug";
        this.descriptionSelector = "description";
        this.metaTitleSelector = "meta_title";
        this.metaDescriptionSelector = "meta_description";
        this.previewSlugSelector = "preview_slug";
        this.previewTitleSelector = "preview_title";
        this.previewDescriptionSelector = "preview_description";
        this.titles = [];
        this.slugs = [];
        this.descriptions = [];
        this.metaTitles = [];
        this.metaDescriptions = [];
        this.previewTitles = [];
        this.previewSlugs = [];
        this.previewDescriptions = [];
        this.descriptionMaxLenght = 150;
        this.titleMaxLenght = 69;
        this.slugMaxLenght = 40;
        this.langCount = $("#language-tabs").children("li").length || 1;

        this.getInputs(this.titleSelector, this.titles);
        this.getInputs(this.slugSelector, this.slugs);
        this.getInputs(this.descriptionSelector, this.descriptions);
        this.getInputs(this.metaTitleSelector, this.metaTitles);
        this.getInputs(this.metaDescriptionSelector, this.metaDescriptions);
        this.getInputs(this.previewSlugSelector, this.previewSlugs);
        this.getInputs(this.previewTitleSelector, this.previewTitles);
        this.getInputs(
            this.previewDescriptionSelector,
            this.previewDescriptions
        );
    }

    getInputs(name, collection) {
        for (let i = 1; i <= this.langCount; i += 1) {
            if (Array.isArray(name)) {
                name.forEach((name) => {
                    const input = $(`#${name}-${i}`);
                    if (input.length) {
                        collection.push(input);
                    }
                });
            } else {
                collection.push($(`#${name}-${i}`));
            }
        }
    }

    strip(html) {
        var tmp = document.createElement("DIV");
        tmp.innerHTML = html;
        return tmp.textContent || tmp.innerText || "";
    }

    convertToSlug(value) {
        value = value
            .toLowerCase()
            // .replace(' ', '-')
            // .replace('[^a-z0-9]+', '');
            .replace(/^\s+|\s+$/g, "")
            .replace(/[_|\s]+/g, "-")
            .replace(/[^a-z\u0400-\u04FF0-9-]+/g, "")
            .replace(/[-]+/g, "-")
            .replace(/^-+|-+$/g, "")
            .replace(/ +/g, "-");

        return value;
    }

    convertToSlugDash(value) {
        value = value
            .toLowerCase()
            .replace(" ", "-")
            .replace(/^\s+|\s+$/g, "")
            .replace(/[_|\s]+/g, "-")
            .replace(/[^a-z\u0400-\u04FF0-9-]+/g, "")
            .replace(/[-]+/g, "-")
            // .replace(/^-+|-+$/g, "")
            .replace(/ +/g, "-")
            .replace("[^a-z0-9]+", "");

        return value;
    }

    transliterate(word) {
        const a = {
            Ё: "YO",
            Й: "Y",
            Ц: "C",
            У: "U",
            К: "K",
            Е: "E",
            Н: "N",
            Г: "G",
            Ш: "SH",
            Щ: "SHT",
            З: "Z",
            Х: "H",
            Ъ: "A",
            ё: "yo",
            й: "y",
            ц: "c",
            у: "u",
            к: "k",
            е: "e",
            н: "n",
            г: "g",
            ш: "sh",
            щ: "sht",
            з: "z",
            х: "h",
            ъ: "a",
            Ф: "F",
            Ы: "I",
            В: "V",
            А: "a",
            П: "P",
            Р: "R",
            О: "O",
            Л: "L",
            Д: "D",
            Ж: "J",
            Э: "E",
            ф: "f",
            ы: "i",
            в: "v",
            а: "a",
            п: "p",
            р: "r",
            о: "o",
            л: "l",
            д: "d",
            ж: "j",
            э: "e",
            Я: "Ya",
            Ч: "CH",
            С: "S",
            М: "M",
            И: "I",
            Т: "T",
            Ь: "Y",
            Б: "B",
            Ю: "YU",
            я: "ya",
            ч: "ch",
            с: "s",
            м: "m",
            и: "i",
            т: "t",
            ь: "y",
            б: "b",
            ю: "yu",
        };
        return word
            .split("")
            .map(function (char) {
                return a[char] || char;
            })
            .join("");
    }

    titleHandler() {
        this.titles.forEach((element, index) => {
            if (element.length === 0) {
                return;
            }

            element = $(element);

            const metaTitle = this.metaTitles[index],
                slug = this.slugs[index];

            let metaTitleSync = false,
                slugSync = false;

            element.focus(() => {
                if (metaTitle.length) {
                    if (element.val() === metaTitle.val().toString()) {
                        metaTitleSync = true;
                    } else {
                        metaTitleSync = false;
                    }
                }

                if (slug.length) {
                    if (element.val() === slug.val().toString()) {
                        slugSync = true;
                    } else {
                        slugSync = false;
                    }
                }
            });

            element.on("keyup", () => {
                const value = element.val(),
                    slugValue = this.transliterate(
                        this.convertToSlugDash(value)
                    );

                if (metaTitleSync) {
                    metaTitle.val(value).trigger("change");
                    this.previewTitles[index].html(value);
                }
                if (slugSync) {
                    slug.val(slugValue).trigger("change");
                    this.previewSlugs[index].html(slugValue);
                }
            });
        });
    }

    descriptionHandler() {
        this.descriptions.forEach((element, index) => {
            if (element.length === 0) {
                return;
            }

            element = $(element);

            const metaDescription = this.metaDescriptions[index],
                previewDescription = this.previewDescriptions[index];

            let metaDescriptionSync = false;

            element.focus(() => {
                if (metaDescription.length) {
                    if (element.val() === metaDescription.val().toString()) {
                        metaDescriptionSync = true;
                    } else {
                        metaDescriptionSync = false;
                    }
                }
            });

            element.on("keyup", () => {
                const value = this.strip(element.val()).substring(0, 150);
                if (metaDescriptionSync && value && value !== "undefined") {
                    metaDescription.val(value).trigger("change");
                    previewDescription.html(value);
                }
            });
        });
    }

    charsLeftColor(charsLeft, closestCounter) {
        if (charsLeft < 0) {
            closestCounter.css("color", Globals.colors.danger);
        } else {
            closestCounter.css("color", Globals.colors.body);
        }
    }

    metaDescriptionHandler() {
        this.metaDescriptions.forEach((element, index) => {
            if (element.length === 0) {
                return;
            }

            element = $(element);

            const closestCounter = element.siblings(".meta_description_length");
            closestCounter.html(
                `${this.descriptionMaxLenght - element.val().length} of ${
                    this.descriptionMaxLenght
                }`
            );

            if (element.val().length) {
                this.previewDescriptions[index].html(element.val());
            }

            element.on("change", () => {
                const value = element.val(),
                    charsLeft = this.descriptionMaxLenght - value.length;
                this.charsLeftColor(charsLeft, closestCounter);
                closestCounter.html(
                    `${charsLeft} of ${this.descriptionMaxLenght}`
                );
            });

            element.on("keyup", () => {
                const value = element.val(),
                    charsLeft = this.descriptionMaxLenght - value.length;

                this.previewDescriptions[index].html(value);
                this.charsLeftColor(charsLeft, closestCounter);
                closestCounter.html(
                    `${charsLeft} of ${this.descriptionMaxLenght}`
                );
            });
        });
    }

    metaTitleHandler() {
        this.metaTitles.forEach((element, index) => {
            if (element.length === 0) {
                return;
            }

            element = $(element);

            const closestCounter = element.siblings(".meta_title_length");
            closestCounter.html(
                `${this.titleMaxLenght - element.val().length} of ${
                    this.titleMaxLenght
                }`
            );

            if (element.val().length) {
                this.previewTitles[index].html(element.val());
            }

            element.on("change", () => {
                const value = element.val(),
                    charsLeft = this.titleMaxLenght - value.length;
                this.charsLeftColor(charsLeft, closestCounter);
                closestCounter.html(`${charsLeft} of ${this.titleMaxLenght}`);
            });

            element.on("keyup", () => {
                const value = element.val(),
                    charsLeft = this.titleMaxLenght - value.length;
                this.previewTitles[index].html(value);
                this.charsLeftColor(charsLeft, closestCounter);
                closestCounter.html(`${charsLeft} of ${this.titleMaxLenght}`);
            });
        });
    }

    slugHandler() {
        this.slugs.forEach((element, index) => {
            if (element.length === 0) {
                return;
            }

            element = $(element);
            const closestCounter = element.siblings(".slug_length");
            closestCounter.html(
                `${this.slugMaxLenght - element.val().length} of ${
                    this.slugMaxLenght
                }`
            );

            if (element.val().length) {
                this.previewSlugs[index].html(element.val());
            }

            element.on("change", () => {
                const value = element.val(),
                    charsLeft = this.slugMaxLenght - value.length;
                this.charsLeftColor(charsLeft, closestCounter);
                closestCounter.html(`${charsLeft} of ${this.slugMaxLenght}`);
            });

            element.on("keyup", () => {
                const value = element.val(),
                    charsLeft = this.slugMaxLenght - value.length,
                    slugifiedValue = this.transliterate(
                        this.convertToSlugDash(value)
                    );

                element.val(slugifiedValue);
                this.previewSlugs[index].html(slugifiedValue);
                this.charsLeftColor(charsLeft, closestCounter);
                closestCounter.html(`${charsLeft} of ${this.slugMaxLenght}`);
            });
        });
    }
}

export default Seo;
