import { CONSTANTS as c, functions as f } from '../helpers';
import Core from '../utilities/Core';
import gsap from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';
gsap.registerPlugin(ScrollTrigger);

import { throttle } from 'lodash-es';

class AnimationController {
  init() {
    const self = this;

    self.goToAnchorTag();

    self.numberCounter();

    self.infinityHorizontalImageScroll();

    self.tabsContent();

    self.scrollUp();

    self.setScrollUpBtnState();

    self.setSectionAnimations();

    self.setBatchAnimations($('[data-load-more] [data-load-item]'));

    $(window).on(
      'scroll',
      throttle(function () {
        self.setScrollUpBtnState();
      })
    );

    // $(window).on(
    //   'resize',
    //   throttle(function () {
    //     ScrollTrigger.refresh();
    //   }, 1)
    // );
    return self;
  }

  goToAnchorTag() {
    const self = this;
    $('[data-anchor]').each(function (index, el) {
      $(el).on('click', function (e) {
        // e.preventDefault();
        setTimeout(() => {
          self.goToTarget(el);
        }, 150);
      });
    });
    $('[data-anchor-target]').each(function (index, el) {
      setTimeout(() => {
        self.goToTarget(el);
      }, 150);
    });

    return self;
  }
  goToTarget(el) {
    let target = window.location.hash;
    if (!target) return;
    target = target.split('#')[1];

    if (!target) return;
    if (!target.length) return;

    let $anchorTarget;
    if ($(el).is('[data-anchor-target]')) {
      $anchorTarget = $(el);
    }
    if ($(el).is('[data-anchor]')) {
      $anchorTarget = $(`[data-anchor-target=${target}]`);
    }

    if (!$anchorTarget.length) return;
    if (target === $anchorTarget.data('anchor-target')) {
      $('html, body').animate(
        {
          scrollTop: $anchorTarget.offset().top,
        },
        100
      );
    }
  }

  setBatchAnimations($els) {
    const self = this;
    if (!$els.length) return;
    let mm = gsap.matchMedia();
    let breakPoint = +c.MQ.lg;

    mm.add(
      {
        // set up any number of arbitrarily-named conditions. The function below will be called when ANY of them match.
        isDesktop: `(min-width: ${breakPoint + 1}px)`,
        isMobile: `(max-width: ${breakPoint}px)`,
        reduceMotion: '(prefers-reduced-motion: reduce)',
      },
      (context) => {
        // context.conditions has a boolean property for each condition defined above indicating if it's matched or not.
        let { isDesktop, isMobile, reduceMotion } = context.conditions;

        if (isDesktop) {
          Core.isDev && console.log('🚀 ~ AnimationController ~ init ~ isDesktop:', isDesktop);
          ScrollTrigger.batch($els, {
            interval: 0.3, // time window (in seconds) for batching to occur.
            batchMax: self.batchMax, // maximum batch size (targets)
            toggleActions: 'play none none none',
            onEnter: (batch) => {
              gsap.to(batch, { autoAlpha: 1, y: 0, stagger: 0.1 });
              batch.forEach((el) => {
                el.dataset.loaded = 'true';
              });
            },
          });
        }
        if (isMobile || reduceMotion) {
          Core.isDev && console.log('🚀 ~ AnimationController ~ init ~ isMobile:', isMobile);
          gsap.set($els, { autoAlpha: 1, y: 0 });
        }

        return () => {
          // optionally return a cleanup function that will be called when none of the conditions match anymore (after having matched)
          // it'll automatically call context.revert() - do NOT do that here . Only put custom cleanup code here.
        };
      }
    );

    return self;
  }

  setSectionAnimations() {
    const self = this;

    let mm = gsap.matchMedia();
    let breakPoint = +c.MQ.lg;

    mm.add(
      {
        // set up any number of arbitrarily-named conditions. The function below will be called when ANY of them match.
        isDesktop: `(min-width: ${breakPoint + 1}px)`,
        isMobile: `(max-width: ${breakPoint}px)`,
        reduceMotion: '(prefers-reduced-motion: reduce)',
      },
      (context) => {
        // context.conditions has a boolean property for each condition defined above indicating if it's matched or not.
        let { isDesktop, isMobile, reduceMotion } = context.conditions;

        if (isDesktop) {
          console.log('🚀 ~ AnimationController ~ init ~ isDesktop:', isDesktop);
          $('section').each(function (index, el) {
            const $el = $(el);

            if ($el.isInViewport() || $el.is('[data-no-section-animation]')) {
              return;
            }
            let tl = gsap.timeline({
              paused: true,
              scrollTrigger: {
                trigger: el,
                start: 'top bottom-=10%',
                toggleActions: 'play none none none',
                // scrub: 1,
                // markers: Core.isDev ? true : false,
              },
            });
            if ($el.prev().is('hr')) {
              tl.from(
                $el.prev(),
                {
                  y: 50,
                  autoAlpha: 0,
                  stagger: 0.5,
                },
                'end'
              );
            }
            tl.from(
              el,
              {
                y: 50,
                autoAlpha: 0,
                stagger: 0.5,
                delay: $el.prev().is('hr') ? 0.5 : 0,
              },
              0
            );
          });
        }
        if (isMobile || reduceMotion) {
        }

        return () => {
          // optionally return a cleanup function that will be called when none of the conditions match anymore (after having matched)
          // it'll automatically call context.revert() - do NOT do that here . Only put custom cleanup code here.
        };
      }
    );

    return self;
  }
  scrollUp() {
    const self = this;
    $('[data-scroll-up]').on('click', function (e) {
      e.preventDefault();
      window.scrollTo(0, 0);
    });
    return self;
  }
  setScrollUpBtnState() {
    const self = this;
    if ($(document).scrollTop() > 0.1 * $(document).height()) {
      $('[data-scroll-up]').removeClass('hidden');
    } else {
      $('[data-scroll-up]').addClass('hidden');
    }
    return self;
  }
  tabsContent() {
    const self = this;
    if (!$('[data-tab-content]').length) return;
    $('[data-tab-content]').each(function (index, el) {
      const $section = $(el);
      $section.find('[data-tab]').each(function (index, el) {
        const $tab = $(el);
        const $content = $section.find(`[data-content=${$tab.attr('data-tab')}]`);
        const $allContents = $section.find('[data-content]');
        const $allTabs = $section.find('[data-tab]');
        $tab.on('click', function (e) {
          e.preventDefault();
          if (c.MQ_IS.xl) {
            $.magnificPopup.open({
              items: {
                src: $content,
              },
              showCloseBtn: false,
              callbacks: {
                open: function () {
                  f.onMagnificOpenCb($content);
                },
                close: function () {
                  f.onMagnificCloseCb($content);
                },
              },
            });
          } else {
            $allContents.removeClass('active');
            $allTabs.removeClass('active');
            $content.addClass('active');
            $tab.addClass('active');
          }
        });
      });
    });
    return self;
  }
  numberCounter() {
    const self = this;
    $('[data-counter]').each(function (index, el) {
      const $el = $(el);
      const $plus = $el.siblings('span');
      let tl = gsap.timeline({
        paused: true,
        scrollTrigger: {
          trigger: el,
          start: 'top bottom-=10%',
          toggleActions: 'play none none none',
          // scrub: 1,
          // markers: Core.isDev ? true : false,
        },
      });
      tl.to($el, {
        autoAlpha: 1,
        duration: 0.5,
        ease: 'linear',
      });
      tl.counter($el, { end: +$el.text(), increment: 5, duration: 0.6 }, '-=0.5');
      tl.to($plus, {
        autoAlpha: 1,
        duration: 0.5,
        ease: 'linear',
      });
    });
    return self;
  }
  infinityHorizontalImageScroll() {
    const sections = document.querySelectorAll('[data-horizontal-scroll]');
    sections.forEach((section) => {
      // add data-animated="true" to every `[data-horizontal-scroll]` on the page
      if (!window.matchMedia('(prefers-reduced-motion: reduce)').matches) {
        section.setAttribute('data-animated', true);
      } else {
        section.setAttribute('data-animated', false);
      }
      if (!section.dataset.animated) return;

      let iteration = 0;
      let scroller = section.querySelector('.flex.items-center');
      console.log("🚀 ~ AnimationController ~ sections.forEach ~ scroller:", scroller)
      if (!scroller) return;
      let scrollerChildren = Array.from(scroller.children);
      if (!scrollerChildren.length) return;

      while (
        scroller.offsetWidth < 3840 * 2 &&
        iteration < 50 &&
        scrollerChildren.length < 200
      ) {
        // Make an array from the elements within `.flex`
        scrollerChildren = Array.from(scroller.children);

        // For each item in the array, clone it
        // add aria-hidden to it
        // add it into the `.flex`
        scrollerChildren.forEach((item) => {
          const duplicatedItem = item.cloneNode(true);
          duplicatedItem.setAttribute('aria-hidden', true);
          scroller.appendChild(duplicatedItem);
        });
        iteration += 1;
      }
    });
  }
}
export const controller = new AnimationController();
