<?php

namespace App\Models\Sections;

use App\Forms\InputFieldNames;
use Kalnoy\Nestedset\NodeTrait;
use App\Models\Sections\Section;
use App\Models\Sections\SectElType;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SectionElement extends Model
{
    use NodeTrait;

    protected $guarded = ['id', 'created_at', 'updated_at'];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    public function translate(string $locale): SectionElement
    {
        $this->locale = $locale;
        return $this;
    }

    private function getFieldDataType(string $fieldName): ?string
    {
        return match ($fieldName) {
            InputFieldNames::TITLE,
            InputFieldNames::SUBTITLE,
            InputFieldNames::AUTHOR,
            InputFieldNames::POSITION,
            InputFieldNames::LINK_TEXT,
            InputFieldNames::LINK,
            InputFieldNames::BUTTON_TEXT,
            InputFieldNames::BUTTON_LINK,
            InputFieldNames::BUTTON_RIGHT_TEXT,
            InputFieldNames::BUTTON_RIGHT_LINK,
            InputFieldNames::IMAGE,
            InputFieldNames::BIG_IMAGE,
            InputFieldNames::SMALL_IMAGE,
            InputFieldNames::BACKGROUND_IMAGE => 'string_value',
            InputFieldNames::TEXT_CONTENT,
            InputFieldNames::DESCRIPTION,
            InputFieldNames::ARTICLE => 'text_value',
            InputFieldNames::NUMBER => 'int_value',
            default => null,
        };
    }

    private function getFieldValue(string $fieldName)
    {
        if (
            !$field = $this->sectElFields->where(
                'sect_el_type_field_id',
                $this->sectElType->fields->firstWhere('name', $fieldName)->id
            )->first()
        ) {
            return null;
        }

        $translation = $field->translate($this->locale ?? app()->getLocale());
        $dataType = $this->getFieldDataType($fieldName);

        return $dataType ? $translation->{$dataType} ?? null : null;
    }

    private function setFieldValue(string $fieldName, $value)
    {
        if (!$dataType = $this->getFieldDataType($fieldName)) {
            return;
        }

        if (!$fieldTypeId = $this->sectElType->fields->firstWhere('name', $fieldName)->id) {
            return;
        }

        if (!$field = $this->sectElFields()->where('sect_el_type_field_id', $fieldTypeId)->first()) {
            $field = $this->sectElFields()->create(['sect_el_type_field_id' => $fieldTypeId]);
        }

        $translation = $field->translateOrNew($this->locale ?? app()->getLocale());
        $translation->$dataType = $value;
        $field->save();
    }

    protected function title(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::TITLE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::TITLE, $value),
        );
    }

    protected function subtitle(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::SUBTITLE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::SUBTITLE, $value),
        );
    }

    protected function author(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::AUTHOR),
            set: fn($value) => $this->setFieldValue(InputFieldNames::AUTHOR, $value),
        );
    }

    protected function position(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::POSITION),
            set: fn($value) => $this->setFieldValue(InputFieldNames::POSITION, $value),
        );
    }

    protected function textContent(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::TEXT_CONTENT),
            set: fn($value) => $this->setFieldValue(InputFieldNames::TEXT_CONTENT, $value),
        );
    }

    protected function description(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::DESCRIPTION),
            set: fn($value) => $this->setFieldValue(InputFieldNames::DESCRIPTION, $value),
        );
    }

    protected function linkText(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::LINK_TEXT),
            set: fn($value) => $this->setFieldValue(InputFieldNames::LINK_TEXT, $value),
        );
    }

    protected function link(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::LINK),
            set: fn($value) => $this->setFieldValue(InputFieldNames::LINK, $value),
        );
    }

    protected function image(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::IMAGE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::IMAGE, $value),
        );
    }

    protected function bigImage(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BIG_IMAGE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BIG_IMAGE, $value),
        );
    }

    protected function smallImage(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::SMALL_IMAGE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::SMALL_IMAGE, $value),
        );
    }

    protected function backgroundImage(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BACKGROUND_IMAGE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BACKGROUND_IMAGE, $value),
        );
    }

    protected function article(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::ARTICLE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::ARTICLE, $value),
        );
    }

    protected function buttonText(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BUTTON_TEXT),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BUTTON_TEXT, $value),
        );
    }

    protected function buttonLink(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BUTTON_LINK),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BUTTON_LINK, $value),
        );
    }

    protected function buttonRightText(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BUTTON_RIGHT_TEXT),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BUTTON_RIGHT_TEXT, $value),
        );
    }

    protected function buttonRightLink(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BUTTON_RIGHT_LINK),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BUTTON_RIGHT_LINK, $value),
        );
    }

    protected function number(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::NUMBER),
            set: fn($value) => $this->setFieldValue(InputFieldNames::NUMBER, $value),
        );
    }

    public function section(): BelongsTo
    {
        return $this->belongsTo(Section::class);
    }

    public function sectElFields(): HasMany
    {
        return $this->hasMany(SectElField::class);
    }

    public function sectElType(): BelongsTo
    {
        return $this->belongsTo(SectElType::class);
    }
}
