<?php

namespace App\Models\Sections;

use App\Models\Image;
use App\Forms\InputFieldNames;

use Kalnoy\Nestedset\NodeTrait;
use App\Models\Sections\SectionType;
use App\Models\Sections\SectionField;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\MorphToMany;

class Section extends Model
{
    use HasFactory, NodeTrait;

    private $locale;

    protected $guarded = ['id'];

    public function sectionable(): MorphTo
    {
        return $this->morphTo();
    }

    public function translate(string $locale): Section
    {
        $this->locale = $locale;
        return $this;
    }

    public function typeHasTitle(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::TITLE)->isNotEmpty();
    }

    public function typeHasSubtitle(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::SUBTITLE)->isNotEmpty();
    }

    public function typeHasDescription(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::DESCRIPTION)->isNotEmpty();
    }

    public function typeHasColor(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::COLOR)->isNotEmpty();
    }

    public function typeHasImage(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::IMAGE)->isNotEmpty();
    }

    public function typeHasVideo(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::VIDEO)->isNotEmpty();
    }

    public function typeHasTitleImage(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::TITLE_IMAGE)->isNotEmpty();
    }

    public function typeHasBackgroundImage(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::BACKGROUND_IMAGE)->isNotEmpty();
    }

    public function typeHasLink(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::LINK)->isNotEmpty();
    }

    public function typeHasSearchTitle(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::SEARCH_TITLE)->isNotEmpty();
    }

    public function typeHasHeaderBlockTitle(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::HEADER_BLOCK_TITLE)->isNotEmpty();
    }

    public function typeHasHeaderBlockSubtitle(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::HEADER_BLOCK_SUBTITLE)->isNotEmpty();
    }

    public function typeHasTextContent(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::TEXT_CONTENT)->isNotEmpty();
    }

    public function typeHasLeftColumnTextContent(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::LEFT_COLUMN_TEXT_CONTENT)->isNotEmpty();
    }

    public function typeHasRightColumnTextContent(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::RIGHT_COLUMN_TEXT_CONTENT)->isNotEmpty();
    }

    public function typeHasOrganization(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::ORGANIZATION)->isNotEmpty();
    }

    public function typeHasIndustry(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::INDUSTRY)->isNotEmpty();
    }

    public function typeHasSolution(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::SOLUTION)->isNotEmpty();
    }

    public function typeHasProducts(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::PRODUCTS)->isNotEmpty();
    }

    public function typeHasLocation(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::LOCATION)->isNotEmpty();
    }

    public function typeHasWebsite(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::WEBSITE)->isNotEmpty();
    }

    public function typeHasAuthor(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::AUTHOR)->isNotEmpty();
    }

    public function typeHasPosition(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::POSITION)->isNotEmpty();
    }

    public function typeHasButtonText(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::BUTTON_TEXT)->isNotEmpty();
    }

    public function typeHasButtonLink(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::BUTTON_LINK)->isNotEmpty();
    }

    public function typeHasButtonRightText(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::BUTTON_RIGHT_TEXT)->isNotEmpty();
    }

    public function typeHasButtonRightLink(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::BUTTON_RIGHT_LINK)->isNotEmpty();
    }

    public function typeHasButtonLeftText(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::BUTTON_LEFT_TEXT)->isNotEmpty();
    }

    public function typeHasButtonLeftLink(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::BUTTON_LEFT_LINK)->isNotEmpty();
    }

    public function typeHasButtonCenterText(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::BUTTON_CENTER_TEXT)->isNotEmpty();
    }

    public function typeHasButtonCenterLink(): bool
    {
        return $this->sectionType->sectionTypeFields->where('name', InputFieldNames::BUTTON_CENTER_LINK)->isNotEmpty();
    }

    private function getFieldDataType(string $fieldName): ?string
    {
        return match ($fieldName) {
            InputFieldNames::COLOR => 'int_value',
            InputFieldNames::LINK,
            InputFieldNames::BUTTON_TEXT,
            InputFieldNames::BUTTON_LINK,
            InputFieldNames::BUTTON_RIGHT_TEXT,
            InputFieldNames::BUTTON_RIGHT_LINK,
            InputFieldNames::BUTTON_LEFT_TEXT,
            InputFieldNames::BUTTON_LEFT_LINK,
            InputFieldNames::BUTTON_CENTER_TEXT,
            InputFieldNames::BUTTON_CENTER_LINK,
            InputFieldNames::TITLE,
            InputFieldNames::SUBTITLE,
            InputFieldNames::IMAGE,
            InputFieldNames::TITLE_IMAGE,
            InputFieldNames::BACKGROUND_IMAGE,
            InputFieldNames::VIDEO,
            InputFieldNames::AUTHOR,
            InputFieldNames::POSITION,
            InputFieldNames::SEARCH_TITLE,
            InputFieldNames::HEADER_BLOCK_TITLE,
            InputFieldNames::HEADER_BLOCK_SUBTITLE,
            InputFieldNames::ORGANIZATION,
            InputFieldNames::INDUSTRY,
            InputFieldNames::SOLUTION,
            InputFieldNames::PRODUCTS,
            InputFieldNames::LOCATION,
            InputFieldNames::WEBSITE => 'string_value',
            InputFieldNames::DESCRIPTION,
            InputFieldNames::TEXT_CONTENT,
            InputFieldNames::LEFT_COLUMN_TEXT_CONTENT,
            InputFieldNames::RIGHT_COLUMN_TEXT_CONTENT => 'text_value',
            default => null,
        };
    }

    private function getFieldValue(string $fieldName)
    {
        if (
            !$field = $this->sectionFields->where(
                'section_type_field_id',
                $this->sectionType->sectionTypeFields->firstWhere('name', $fieldName)->id
            )->first()
        ) {
            return null;
        }

        $translation = $field->translate($this->locale ?? app()->getLocale());
        $dataType = $this->getFieldDataType($fieldName);

        return $dataType ? $translation->{$dataType} ?? null : null;
    }

    private function setFieldValue(string $fieldName, $value)
    {
        if (!$dataType = $this->getFieldDataType($fieldName)) {
            return;
        }

        if (!$fieldTypeId = $this->sectionType->sectionTypeFields->firstWhere('name', $fieldName)->id) {
            return;
        }

        if (!$field = $this->sectionFields()->where('section_type_field_id', $fieldTypeId)->first()) {
            $field = $this->sectionFields()->create(['section_type_field_id' => $fieldTypeId]);
        }

        $translation = $field->translateOrNew($this->locale ?? app()->getLocale());
        $translation->$dataType = $value;
        $field->save();
    }

    protected function title(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::TITLE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::TITLE, $value),
        );
    }

    protected function subtitle(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::SUBTITLE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::SUBTITLE, $value),
        );
    }

    protected function description(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::DESCRIPTION),
            set: fn($value) => $this->setFieldValue(InputFieldNames::DESCRIPTION, $value),
        );
    }

    protected function color(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::COLOR),
            set: fn($value) => $this->setFieldValue(InputFieldNames::COLOR, $value),
        );
    }

    protected function image(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::IMAGE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::IMAGE, $value),
        );
    }

    protected function titleImage(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::TITLE_IMAGE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::TITLE_IMAGE, $value),
        );
    }

    protected function backgroundImage(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BACKGROUND_IMAGE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BACKGROUND_IMAGE, $value),
        );
    }

    protected function video(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::VIDEO),
            set: fn($value) => $this->setFieldValue(InputFieldNames::VIDEO, $value),
        );
    }

    protected function link(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::LINK),
            set: fn($value) => $this->setFieldValue(InputFieldNames::LINK, $value),
        );
    }

    protected function buttonText(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BUTTON_TEXT),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BUTTON_TEXT, $value),
        );
    }

    protected function buttonLink(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BUTTON_LINK),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BUTTON_LINK, $value),
        );
    }

    protected function buttonRightText(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BUTTON_RIGHT_TEXT),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BUTTON_RIGHT_TEXT, $value),
        );
    }

    protected function buttonRightLink(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BUTTON_RIGHT_LINK),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BUTTON_RIGHT_LINK, $value),
        );
    }

    protected function buttonLeftText(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BUTTON_LEFT_TEXT),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BUTTON_LEFT_TEXT, $value),
        );
    }

    protected function buttonLeftLink(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BUTTON_LEFT_LINK),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BUTTON_LEFT_LINK, $value),
        );
    }

    protected function buttonCenterText(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BUTTON_CENTER_TEXT),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BUTTON_CENTER_TEXT, $value),
        );
    }

    protected function buttonCenterLink(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::BUTTON_CENTER_LINK),
            set: fn($value) => $this->setFieldValue(InputFieldNames::BUTTON_CENTER_LINK, $value),
        );
    }

    protected function searchTitle(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::SEARCH_TITLE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::SEARCH_TITLE, $value),
        );
    }

    protected function headerBlockTitle(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::HEADER_BLOCK_TITLE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::HEADER_BLOCK_TITLE, $value),
        );
    }

    protected function headerBlockSubtitle(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::HEADER_BLOCK_SUBTITLE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::HEADER_BLOCK_SUBTITLE, $value),
        );
    }

    protected function textContent(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::TEXT_CONTENT),
            set: fn($value) => $this->setFieldValue(InputFieldNames::TEXT_CONTENT, $value),
        );
    }

    protected function leftColumnTextContent(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::LEFT_COLUMN_TEXT_CONTENT),
            set: fn($value) => $this->setFieldValue(InputFieldNames::LEFT_COLUMN_TEXT_CONTENT, $value),
        );
    }

    protected function rightColumnTextContent(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::RIGHT_COLUMN_TEXT_CONTENT),
            set: fn($value) => $this->setFieldValue(InputFieldNames::RIGHT_COLUMN_TEXT_CONTENT, $value),
        );
    }

    protected function organization(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::ORGANIZATION),
            set: fn($value) => $this->setFieldValue(InputFieldNames::ORGANIZATION, $value),
        );
    }

    protected function industry(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::INDUSTRY),
            set: fn($value) => $this->setFieldValue(InputFieldNames::INDUSTRY, $value),
        );
    }

    protected function solution(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::SOLUTION),
            set: fn($value) => $this->setFieldValue(InputFieldNames::SOLUTION, $value),
        );
    }

    protected function products(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::PRODUCTS),
            set: fn($value) => $this->setFieldValue(InputFieldNames::PRODUCTS, $value),
        );
    }

    protected function location(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::LOCATION),
            set: fn($value) => $this->setFieldValue(InputFieldNames::LOCATION, $value),
        );
    }

    protected function website(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::WEBSITE),
            set: fn($value) => $this->setFieldValue(InputFieldNames::WEBSITE, $value),
        );
    }

    protected function author(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::AUTHOR),
            set: fn($value) => $this->setFieldValue(InputFieldNames::AUTHOR, $value),
        );
    }

    protected function position(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->getFieldValue(InputFieldNames::POSITION),
            set: fn($value) => $this->setFieldValue(InputFieldNames::POSITION, $value),
        );
    }

    public function scopeFilterByParentIdAndType(Builder $query, int $sectionableId, string $class): void
    {
        $query
            ->select('id', 'section_type_id', 'cover_image', 'parent_id', '_lft', '_rgt', 'is_active')
            ->where('sectionable_id', $sectionableId)
            ->where('sectionable_type', $class)
            ->with('sectionType')
            ->with('sectionType.sectionTypeFields')
            ->with('sectionFields:id,section_id,section_type_field_id')
            ->with('sectionFields.translations');
    }

    public function sectionFields(): HasMany
    {
        return $this->hasMany(SectionField::class);
    }

    public function sectionType(): BelongsTo
    {
        return $this->belongsTo(SectionType::class);
    }

    public function images(): MorphToMany
    {
        return $this->morphToMany(Image::class, 'imageable');
    }

    public function imagesOrdered(): Collection
    {
        return $this->morphToMany(Image::class, 'imageable')->orderBy('order')->get();
    }

    public function sectionElements(): HasMany
    {
        return $this->hasMany(SectionElement::class);
    }
}
