<?php

namespace App\Http\Requests\Admin;

use Illuminate\Validation\Rule;
use Illuminate\Foundation\Http\FormRequest;
use Astrotomic\Translatable\Validation\RuleFactory;

class PageRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array|string>
     */
    public function rules(): array
    {
        if ($this->method() === 'PATCH') {
            $rules = [
                'is_active' => ['in:0,1'],
                'is_favorite' => ['in:0,1'],
            ];
        } else {
            $rules = [
                'page_type_id' => ['required', 'exists:page_types,id'],
                'cover_image' => ['sometimes', 'nullable', 'string', 'max:255'],
                'is_active' => ['in:0,1'],
                'is_favorite' => ['in:0,1'],

                '%slug%' => ['required', 'string', 'min:3', 'max:50'],
                '%title%' => ['required', 'string', 'min:3', 'max:255'],
                '%description%' => ['max:255'],
                '%meta_title%' => ['sometimes', 'nullable', 'string', 'min:3', 'max:70'],
                '%meta_description%' => ['max:160'],
            ];

            if (!empty($this->page->id)) {
                $rules['%slug%'][] = Rule::unique('page_translations', 'slug')
                    ->ignore($this->page->id, 'page_id')
                    ->where('page_id', $this->page->id);
            } else {
                $rules['%slug%'][] = 'unique:page_translations,slug';
            }
        }

        return RuleFactory::make($rules);
    }

    public function prepareForValidation()
    {
        foreach ($this->all() as $lang => $fields) {
            if (is_array($fields)) {
                if (isset($fields['slug']) && strlen($fields['slug']) > 49) {
                    $fields['slug'] = mb_substr($fields['slug'], 0, 49);
                }

                if (isset($fields['meta_title']) && strlen($fields['meta_title']) > 69) {
                    $fields['meta_title'] = substr($fields['meta_title'], 0, 69);
                }

                if (isset($fields['meta_description']) && strlen($fields['meta_description']) > 159) {
                    $fields['meta_description'] = mb_substr($fields['meta_description'], 0, 159);
                }

                $this->merge([$lang => $fields]);
            }
        }
    }
}
