<?php

namespace App\Http\Controllers;

use App\Models\Menu;
use App\Models\Page;
use App\Models\Domain;
use Illuminate\View\View;
use Illuminate\Http\Request;
use App\Services\MenuService;
use Illuminate\Http\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;

class PageController extends Controller
{
    private array $menuTypes;

    public function __construct(private MenuService $menuService, private Menu $mainMenu, private Menu $fastLinksMenu, private Domain $domain)
    {
        $menus = $menuService();
        $this->menuTypes = $menus['menuTypes'];
        if ($menus['mainMenu']) {
            $this->mainMenu = $menus['mainMenu'];
        }
        if ($menus['fastLinksMenu']) {
            $this->fastLinksMenu = $menus['fastLinksMenu'];
        }
        $this->domain = Domain::first();
    }

    public function home(): View
    {
        $page = Page::whereTranslation('slug', 'home')->where('is_active', true)->pageWithFullContent()->firstOrFail();
        $page->setRelation('sections', $page->sections->toTree());

        return view('front.pages.index', [
            'page' => $page,
            'mainMenu' => $this->mainMenu ?? null,
            'fastLinksMenu' => $this->fastLinksMenu ?? null,
            'menuTypes' => $this->menuTypes,
            'domain' => $this->domain,
            'slug' => 'home',
            'urlParentSlug' => null,
        ]);
    }

    public function show(string $url): View|RedirectResponse
    {
        if ($url === 'home') {
            return redirect()->route('home', [], Response::HTTP_MOVED_PERMANENTLY);
        }
        $slug = basename($url);
        $page = Page::whereTranslation('slug', $slug)->pageWithFullContent()->where('is_active', true)->firstOrFail();

        $urlParentSlug = get_directory_part($url) ?? null;
        if ($urlParentSlug && !$page->parent_id || !$urlParentSlug && $page->parent_id) {
            return abort(Response::HTTP_NOT_FOUND);
        }
        if ($urlParentSlug && $page->parent_id) {
            $parentPage = Page::without('translations')->where('id', $page->parent_id)->first();
            if ($parentPage->slug !== $urlParentSlug) {
                return abort(Response::HTTP_NOT_FOUND);
            }
        }

        $page->setRelation('sections', $page->sections->toTree());

        $breadcrumbs = [
            [
                'name' => __('Home'),
                'is_last' => false,
                'link' => route('home'),
            ]
        ];
        if ($page->parent_id) {
            $breadcrumbs[] = [
                'name' => $page->parent->title,
                'is_last' => false,
                'link' => route('pages.show', $urlParentSlug),
            ];
        }

        $breadcrumbs[] = [
            'name' => $page->title,
            'is_last' => true,
            'link' => route('pages.show', $page->slug),
        ];

        return view('front.pages.show', [
            'page' => $page,
            'mainMenu' => $this->mainMenu ?? null,
            'fastLinksMenu' => $this->fastLinksMenu ?? null,
            'menuTypes' => $this->menuTypes,
            'domain' => $this->domain,
            'breadcrumbs' => $breadcrumbs,
            'slug' => $slug,
            'urlParentSlug' => $urlParentSlug,
        ]);
    }

    public function search(Request $request): View
    {
        $search = $request->input('search');
        $pages = Page::when($search, fn($query, $search) => $query->filterTranslations($search))->paginate(10)->withQueryString();

        $breadcrumbs = [
            [
                'name' => __('Home'),
                'is_last' => false,
                'link' => route('home'),
            ],
            [
                'name' => __('Search'),
                'is_last' => true,
                'link' => route('pages.search'),
            ]
        ];

        return view('front.pages.search', [
            'page' => Page::whereTranslation('slug', 'search')->where('is_active', true)->firstOrFail(),
            'pages' => $pages,
            'mainMenu' => $this->mainMenu ?? null,
            'fastLinksMenu' => $this->fastLinksMenu ?? null,
            'menuTypes' => $this->menuTypes,
            'domain' => $this->domain,
            'breadcrumbs' => $breadcrumbs,
            'slug' => 'search',
            'urlParentSlug' => null,
        ]);
    }
}
