<?php

namespace App\Http\Controllers\Admin;

use App\Models\Page;
use App\Models\PageType;
use App\Traits\Sortable;
use Illuminate\View\View;
use App\Services\ToggleSwitch;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\Admin\SolutionRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Controllers\Admin\PanelController;

class SolutionController extends PanelController
{
    use Sortable;

    const SEARCH_ROUTE = 'admin.pages.index';

    public function __construct()
    {
        $this->model = new Page();
    }

    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        return view('admin.solutions.index', [
            'solutions' => Page::when(request('search'), fn($query, $search) => $query->filterTranslations($search))
                ->where('page_type_id', PageType::where('name', 'solutions')->first()->id)
                ->latest()
                ->get()
                ->toTree(),
            'languages' => languages(),
            'searchRoute' => self::SEARCH_ROUTE,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.solutions.create', [
            'languages' => languages(),
            'pageTypes' => PageType::all(),
            'searchRoute' => self::SEARCH_ROUTE,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(SolutionRequest $request): RedirectResponse
    {
        $solution = Page::create($request->except(['_token', '_method', 'save', 'save_close']));

        if ($request->save_close) {
            return to_route('admin.solutions.index')->withSuccess('The solution was created.');
        }

        return to_route('admin.solutions.edit', $solution)->withSuccess('The solution was created.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Page $solution): View
    {
        $solutionParent = Page::find($solution->parent_id);

        return view('admin.solutions.edit', [
            'solution' => $solution,
            'solutionParent' => $solutionParent,
            'pageTypes' => PageType::all(),
        ]);
    }

    public function toggleActive(SolutionRequest $request, Page $solution, ToggleSwitch $toggleSwitch): JsonResponse
    {
        if (!$toggleSwitch($request, $solution, 'is_active')) {
            return response()->json(['error' => 'Somehting went wrong.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
        return response()->json(['success' => 'Record changed', Response::HTTP_OK]);
    }

    public function toggleFavorite(SolutionRequest $request, Page $solution, ToggleSwitch $toggleSwitch): JsonResponse
    {
        if (!$toggleSwitch($request, $solution, 'is_favorite')) {
            return response()->json(['error' => 'Somehting went wrong.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
        return response()->json(['success' => 'Record changed', Response::HTTP_OK]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(SolutionRequest $request, Page $solution): RedirectResponse
    {
        if ($request->cover_image) {
            $this->createImagePreview($request->cover_image);
        }

        if (!$solution->update($request->except(['_token', '_method', 'save', 'save_close']))) {
            return back()->withError('The changes were not saved. Please try again.');
        }

        if ($request->save_close) {
            return to_route('admin.solutions.index')->withSuccess('The changes were saved.');
        }

        return back()->withSuccess('The changes were saved.');
    }
}
