<?php

namespace App\Http\Controllers\Admin;

use App\Traits\Sortable;
use Illuminate\View\View;
use App\Models\Sections\Section;
use Illuminate\Http\JsonResponse;
use App\Models\Sections\SectElType;
use Illuminate\Http\RedirectResponse;
use App\Models\Sections\SectionElement;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Controllers\Admin\PanelController;
use App\Http\Requests\Admin\SectionElementRequest;

class SectionElementController extends PanelController
{
    use Sortable;

    public function __construct()
    {
        $this->model = new SectionElement();
    }

    public function selectType(Section $section): View
    {
        return view('admin.sections.elements.select-type', [
            'sectionElementTypes' => SectElType::all(),
            'section'             => $section,
        ]);
    }

    public function create(): View
    {
        $section = Section::with('sectionType.sectElType')->findOrFail(request()->section);

        return view('admin.sections.elements.create', [
            'languages'                 => languages(),
            'sectionElementType'        => $section->sectionType->sectElType,
            'section'                   => $section,
        ]);
    }

    public function store(SectionElementRequest $request): RedirectResponse
    {
        $section = Section::findOrFail($request->section_id);
        $sectionElement = $section->sectionElements()->create([
            'sect_el_type_id' => $request->section_element_type_id,
            ...$request->except(['_token', '_method', 'save', 'save_close', 'fields'])
        ]);

        foreach ($request->fields as $langKey => $langFields) {
            foreach ($langFields as $fieldKey => $langField) {
                if ($langField) {
                    $sectionElement->translate($langKey)->$fieldKey = $langField;
                }
            }
        }

        return to_route('admin.sections.edit', $section)->withSuccess('The section element was created.');
    }

    public function edit(SectionElement $sectionElement): View
    {
        $sectionElement->load('sectElFields.translations', 'sectElFields.translations');
        return view('admin.sections.elements.edit', [
            'languages'             => languages(),
            'sectionElement'        => $sectionElement,
            'sectionElementType'    => SectElType::whereId($sectionElement->sect_el_type_id)->with('fields')->firstOrFail(),
        ]);
    }

    public function update(SectionElementRequest $request, SectionElement $sectionElement): RedirectResponse
    {
        if ($request->cover_image) {
            $this->createImagePreview($request->cover_image);
        }

        if (!$sectionElement->update($request->except(['_token', '_method', 'save', 'save_close', 'fields']))) {
            return back()->withError('The changes were not saved. Please try again.');
        }

        foreach ($request->fields as $langKey => $langFields) {
            foreach ($langFields as $fieldKey => $langField) {
                $sectionElement->translate($langKey)->$fieldKey = $langField ?? '';
            }
        }

        if ($request->save_close) {
            return to_route('admin.sections.edit', $sectionElement->section)->withSuccess('The changes were saved.');
        }

        return back()->withSuccess('The changes were saved.');
    }

    public function uploadImage(SectionElement $sectionElement): JsonResponse
    {
        foreach (explode(',', request()->file_path) as $filePath) {
            if (!$sectionElement->images()->create(['path' => $filePath])) {
                return response()->json(['error' => 'Something went wrong'], Response::HTTP_INTERNAL_SERVER_ERROR);
            }
        }

        return response()->json(['success' => 'The images were uploaded'], Response::HTTP_OK);
    }

    public function deleteImage(SectionElement $sectionElement, string $imageType, string $locale): JsonResponse
    {
        $sectionElement->translate($locale)->$imageType = '';
        $sectionElement->refresh();

        if ($sectionElement->translate($locale)->$imageType === '') {
            return response()->json(['success' => 'The image was deleted'], Response::HTTP_OK);
        }

        return response()->json(['error' => 'Something went wrong'], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}
