<?php

namespace App\Http\Controllers\Admin;

use App\Models\PageType;
use App\Traits\Sortable;
use App\Forms\SelectList;
use Illuminate\View\View;
use App\Services\ToggleSwitch;
use App\Models\Sections\Section;
use Illuminate\Http\JsonResponse;
use App\Models\Sections\SectionType;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\Admin\SectionRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Controllers\Admin\PanelController;

class SectionController extends PanelController
{
    use Sortable;

    public function __construct()
    {
        $this->model = new Section();
    }

    public function selectType(string $parentClass, int $parentId): View
    {
        return view('admin.sections.select-type', [
            'parent'        => $parentClass::findOrFail($parentId),
            'sectionTypes'  => SectionType::all(),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        return view('admin.sections.create', [
            'languages'     => languages(),
            'parent'        => request()->parent_type::findOrFail(request()->parent_id),
            'sectionType'   => SectionType::whereId(request()->type)->firstOrFail(),
            'sectionColors' => (new SelectList)->get('sections.colors'),
            'pageTypes'     => PageType::all(),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(SectionRequest $request): RedirectResponse
    {
        $page = $request->sectionable_type::findOrFail($request->sectionable_id);
        $section = $page->sections()->create([
            'section_type_id' => $request->sectionTypeId,
            ...$request->except(['_token', '_method', 'save', 'save_close', 'fields'])
        ]);

        foreach ($request->fields as $langKey => $langFields) {
            foreach ($langFields as $fieldKey => $langField) {
                if ($langField) {
                    $section->translate($langKey)->$fieldKey = $langField;
                }
            }
        }

        if ($request->save_close) {
            return to_route('admin.' . $section->sectionable->getTable() . '.edit', $request->sectionable_id)->withSuccess('The section was created.');
        }

        return to_route('admin.sections.edit', $section)->withSuccess('The section was created.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Section $section): View
    {
        $sectionableItem = $section->sectionable::class::find($section->sectionable_id);
        $sectionableItemParent = $section->sectionable::class::find($sectionableItem->parent_id);

        return view('admin.sections.edit', [
            'languages'             => languages(),
            'section'               => $section->load('sectionElements', 'sectionType.sectElType'),
            'sectionImages'         => $section->imagesOrdered(),
            'sectionColors'         => (new SelectList)->get('sections.colors'),
            'sectionableItemParent' => $sectionableItemParent,
            'pageTypes'             => PageType::all(),
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(SectionRequest $request, Section $section): RedirectResponse
    {
        if ($request->cover_image) {
            $this->createImagePreview($request->cover_image);
        }

        if (!$section->update($request->except(['_token', '_method', 'save', 'save_close', 'fields']))) {
            return back()->withError('The changes were not saved. Please try again.');
        }

        foreach ($request->fields as $langKey => $langFields) {
            foreach ($langFields as $fieldKey => $langField) {
                $section->translate($langKey)->$fieldKey = $langField;
            }
        }

        if ($request->save_close) {
            return to_route('admin.' . $section->sectionable->getTable() . '.edit', $request->sectionable_id)->withSuccess('The changes were saved.');
        }

        return back()->withSuccess('The changes were saved.');
    }

    public function uploadImage(Section $section): JsonResponse
    {
        foreach (explode(',', request()->file_path) as $filePath) {
            if (!$section->images()->create(['path' => $filePath])) {
                return response()->json(['error' => 'Something went wrong'], Response::HTTP_INTERNAL_SERVER_ERROR);
            }
        }

        return response()->json(['success' => 'The images were uploaded'], Response::HTTP_OK);
    }

    public function deleteImage(Section $section, string $locale, string $imageType): JsonResponse
    {
        $section->translate($locale)->$imageType = '';
        $section->refresh();

        if ($section->translate($locale)->$imageType === '') {
            return response()->json(['success' => 'The image was deleted'], Response::HTTP_OK);
        }

        return response()->json(['error' => 'Something went wrong'], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function toggleActive(SectionRequest $request, Section $section, ToggleSwitch $toggleSwitch): JsonResponse
    {
        if (!$toggleSwitch($request, $section, 'is_active')) {
            return response()->json(['error' => 'Somehting went wrong.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
        return response()->json(['success' => 'Record changed', Response::HTTP_OK]);
    }
}
