<?php

namespace App\Http\Controllers\Admin;

use App\Models\Page;
use App\Models\PageType;
use App\Traits\Sortable;
use Illuminate\View\View;
use App\Services\ToggleSwitch;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\Admin\ProjectRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Controllers\Admin\PanelController;

class ProjectController extends PanelController
{
    use Sortable;

    const SEARCH_ROUTE = 'admin.pages.index';

    public function __construct()
    {
        $this->model = new Page();
    }

    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        return view('admin.projects.index', [
            'projects' => Page::when(request('search'), fn($query, $search) => $query->filterTranslations($search))
                ->where('page_type_id', PageType::where('name', 'projects')->first()->id)
                ->latest()
                ->get()
                ->toTree(),
            'languages' => languages(),
            'searchRoute' => self::SEARCH_ROUTE,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.projects.create', [
            'languages' => languages(),
            'pageTypes' => PageType::all(),
            'searchRoute' => self::SEARCH_ROUTE,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ProjectRequest $request): RedirectResponse
    {
        $project = Page::create($request->except(['_token', '_method', 'save', 'save_close']));

        if ($request->save_close) {
            return to_route('admin.projects.index')->withSuccess('The project was created.');
        }

        return to_route('admin.projects.edit', $project)->withSuccess('The project was created.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Page $project): View
    {
        $projectParent = Page::find($project->parent_id);

        return view('admin.projects.edit', [
            'project' => $project,
            'projectParent' => $projectParent,
            'pageTypes' => PageType::all(),
        ]);
    }

    public function toggleActive(ProjectRequest $request, Page $project, ToggleSwitch $toggleSwitch): JsonResponse
    {
        if (!$toggleSwitch($request, $project, 'is_active')) {
            return response()->json(['error' => 'Somehting went wrong.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
        return response()->json(['success' => 'Record changed', Response::HTTP_OK]);
    }

    public function toggleFavorite(ProjectRequest $request, Page $project, ToggleSwitch $toggleSwitch): JsonResponse
    {
        if (!$toggleSwitch($request, $project, 'is_favorite')) {
            return response()->json(['error' => 'Somehting went wrong.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
        return response()->json(['success' => 'Record changed', Response::HTTP_OK]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ProjectRequest $request, Page $project): RedirectResponse
    {
        if ($request->cover_image) {
            $this->createImagePreview($request->cover_image);
        }

        if (!$project->update($request->except(['_token', '_method', 'save', 'save_close']))) {
            return back()->withError('The changes were not saved. Please try again.');
        }

        if ($request->save_close) {
            return to_route('admin.projects.index')->withSuccess('The changes were saved.');
        }

        return back()->withSuccess('The changes were saved.');
    }
}
