<?php

namespace App\Http\Controllers\Admin;

use App\Models\Page;
use App\Models\PageType;
use App\Traits\Sortable;
use Illuminate\View\View;
use Illuminate\Http\Request;
use App\Services\ToggleSwitch;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\Admin\PageRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Controllers\Admin\PanelController;

class PageController extends PanelController
{
    use Sortable;

    const SEARCH_ROUTE = 'admin.pages.index';

    public function __construct()
    {
        $this->model = new Page();
    }

    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        return view('admin.pages.index', [
            'pages' => Page::when(request('search'), fn($query, $search) => $query->filterTranslations($search))
                ->where('page_type_id', PageType::where('name', 'pages')->first()->id)
                ->latest()
                ->get()
                ->toTree(),
            'languages' => languages(),
            'searchRoute' => self::SEARCH_ROUTE,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.pages.create', [
            'languages' => languages(),
            'pageTypes' => PageType::all(),
            'searchRoute' => self::SEARCH_ROUTE,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(PageRequest $request): RedirectResponse
    {
        $page = Page::create($request->except(['_token', '_method', 'save', 'save_close']));

        if ($request->save_close) {
            return to_route('admin.pages.index')->withSuccess('The page was created.');
        }

        return to_route('admin.pages.edit', $page)->withSuccess('The page was created.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Page $page): View
    {
        $pageParent = Page::find($page->parent_id);

        return view('admin.pages.edit', [
            'page' => $page,
            'pageParent' => $pageParent,
            'pageTypes' => PageType::all(),
        ]);
    }

    public function toggleActive(PageRequest $request, Page $page, ToggleSwitch $toggleSwitch): JsonResponse
    {
        if (!$toggleSwitch($request, $page, 'is_active')) {
            return response()->json(['error' => 'Somehting went wrong.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
        return response()->json(['success' => 'Record changed', Response::HTTP_OK]);
    }

    public function toggleFavorite(PageRequest $request, Page $page, ToggleSwitch $toggleSwitch): JsonResponse
    {
        if (!$toggleSwitch($request, $page, 'is_favorite')) {
            return response()->json(['error' => 'Somehting went wrong.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
        return response()->json(['success' => 'Record changed', Response::HTTP_OK]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(PageRequest $request, Page $page): RedirectResponse
    {
        if ($request->cover_image) {
            $this->createImagePreview($request->cover_image);
        }

        if (!$page->update($request->except(['_token', '_method', 'save', 'save_close']))) {
            return back()->withError('The changes were not saved. Please try again.');
        }

        if ($request->save_close) {
            return to_route('admin.pages.index')->withSuccess('The changes were saved.');
        }

        return back()->withSuccess('The changes were saved.');
    }
}
