<?php

namespace App\Http\Controllers\Admin;

use App\Models\Page;
use App\Models\PageType;
use App\Traits\Sortable;
use Illuminate\View\View;
use App\Services\ToggleSwitch;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\Admin\NewsRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Controllers\Admin\PanelController;

class NewsController extends PanelController
{
    use Sortable;

    const SEARCH_ROUTE = 'admin.pages.index';

    public function __construct()
    {
        $this->model = new Page();
    }

    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        return view('admin.news.index', [
            'news' => Page::when(request('search'), fn($query, $search) => $query->filterTranslations($search))
                ->where('page_type_id', PageType::where('name', 'news')->first()->id)
                ->latest()
                ->get()
                ->toTree(),
            'languages' => languages(),
            'searchRoute' => self::SEARCH_ROUTE,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.news.create', [
            'languages' => languages(),
            'pageTypes' => PageType::all(),
            'searchRoute' => self::SEARCH_ROUTE,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(NewsRequest $request): RedirectResponse
    {
        $news = Page::create($request->except(['_token', '_method', 'save', 'save_close']));

        if ($request->save_close) {
            return to_route('admin.news.index')->withSuccess('The news was created.');
        }

        return to_route('admin.news.edit', $news)->withSuccess('The news was created.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Page $news): View
    {
        $newsParent = Page::find($news->parent_id);

        return view('admin.news.edit', [
            'news' => $news,
            'newsParent' => $newsParent,
            'pageTypes' => PageType::all(),
        ]);
    }

    public function toggleActive(NewsRequest $request, Page $news, ToggleSwitch $toggleSwitch): JsonResponse
    {
        if (!$toggleSwitch($request, $news, 'is_active')) {
            return response()->json(['error' => 'Somehting went wrong.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
        return response()->json(['success' => 'Record changed', Response::HTTP_OK]);
    }

    public function toggleFavorite(NewsRequest $request, Page $news, ToggleSwitch $toggleSwitch): JsonResponse
    {
        if (!$toggleSwitch($request, $news, 'is_favorite')) {
            return response()->json(['error' => 'Somehting went wrong.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
        return response()->json(['success' => 'Record changed', Response::HTTP_OK]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(NewsRequest $request, Page $news): RedirectResponse
    {
        if ($request->cover_image) {
            $this->createImagePreview($request->cover_image);
        }

        if (!$news->update($request->except(['_token', '_method', 'save', 'save_close']))) {
            return back()->withError('The changes were not saved. Please try again.');
        }

        if ($request->save_close) {
            return to_route('admin.news.index')->withSuccess('The changes were saved.');
        }

        return back()->withSuccess('The changes were saved.');
    }
}
