<?php

namespace App\Http\Controllers\Admin;

use App\Models\Menu;
use App\Traits\Sortable;
use App\Forms\SelectList;
use Illuminate\View\View;
use App\Services\ToggleSwitch;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\Admin\MenuRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Controllers\Admin\PanelController;

class MenuController extends PanelController
{
    use Sortable;

    const SEARCH_ROUTE = 'admin.settings.menus.index';

    public function __construct()
    {
        $this->model = new Menu();
    }

    public function index(): View
    {
        return view('admin.settings.menus.index', [
            'menus' => Menu::when(request('search'), fn($query, $search) => $query->filterMenus($search))
                ->defaultOrder()
                ->get()
                ->toTree(),
            'searchRoute' => self::SEARCH_ROUTE,
        ]);
    }

    public function create(): View
    {
        return view('admin.settings.menus.create', [
            'searchRoute' => self::SEARCH_ROUTE,
            'menuTypes' => (new SelectList)->get('menus.types'),
            'languages' => languages(),
        ]);
    }

    public function store(MenuRequest $request): RedirectResponse
    {
        if (!$menu = Menu::create($request->except(['_token', '_method', 'save', 'save_close']))) {
            return back()->withErrors('Something went wrong.');
        }

        if ($request->save_close) {
            return to_route('admin.settings.menus.index')->withSuccess('The menu was created.');
        }

        return to_route('admin.settings.menus.edit', $menu)->withSuccess('The menu was created.');
    }


    public function edit(Menu $menu): View
    {
        return view('admin.settings.menus.edit', [
            'menu' => $menu,
            'menuItems' => $menu->menuItems()
                ->when(request('search'), fn($query, $search) => $query->filterMenuItems($search))
                ->defaultOrder()
                ->get()
                ->toTree(),
            'searchRoute' => ['route' => 'admin.settings.menus.edit', 'params' => ['menu' => $menu->id]],
            'menuTypes' => (new SelectList)->get('menus.types'),
            'languages' => languages(),
        ]);
    }

    public function toggleActive(MenuRequest $request, Menu $menu, ToggleSwitch $toggleSwitch): JsonResponse
    {
        if (!$toggleSwitch($request, $menu, 'is_active')) {
            return response()->json(['error' => 'Somehting went wrong.'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
        return response()->json(['success' => 'Record changed', Response::HTTP_OK]);
    }

    public function update(MenuRequest $request, Menu $menu): RedirectResponse
    {
        if (!$menu->update($request->except(['_token', '_method', 'save', 'save_close']))) {
            return back()->withErrors('The changes were not saved. Please try again.');
        }

        if ($request->save_close) {
            return to_route('admin.settings.menus.index')->withSuccess('The changes were saved.');
        }

        return back()->withSuccess('The changes were saved.');
    }
}
